'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { z } from 'zod';

import { Button } from '@/components/ui/button';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { useToast } from '@/hooks/use-toast';
import { useLanguage } from '@/contexts/language-context';

export function ContactForm() {
  const { toast } = useToast();
  const { content, language } = useLanguage();
  
  const formSchema = z.object({
    name: z.string().min(2, { message: language === 'en' ? 'Name must be at least 2 characters.' : 'يجب أن يتكون الاسم من حرفين على الأقل.' }),
    email: z.string().email({ message: language === 'en' ? 'Invalid email address.' : 'عنوان البريد الإلكتروني غير صالح.' }),
    company: z.string().optional(),
    message: z.string().min(10, { message: language === 'en' ? 'Message must be at least 10 characters.' : 'يجب أن تتكون الرسالة من 10 أحرف على الأقل.' }),
  });


  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      name: '',
      email: '',
      company: '',
      message: '',
    },
  });

  function onSubmit(values: z.infer<typeof formSchema>) {
    console.log(values);
    toast({
      title: language === 'en' ? 'Message Sent!' : 'تم إرسال الرسالة!',
      description: language === 'en' ? "Thank you for reaching out. We'll be in touch soon." : 'شكرًا لتواصلك معنا. سنتواصل معك قريبًا.',
    });
    form.reset();
  }

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
        <FormField
          control={form.control}
          name="name"
          render={({ field }) => (
            <FormItem>
              <FormLabel>{content.contact.form.name}</FormLabel>
              <FormControl>
                <Input placeholder={content.contact.form.name} {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="email"
          render={({ field }) => (
            <FormItem>
              <FormLabel>{content.contact.form.email}</FormLabel>
              <FormControl>
                <Input placeholder={content.contact.form.email} {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="company"
          render={({ field }) => (
            <FormItem>
              <FormLabel>{content.contact.form.company}</FormLabel>
              <FormControl>
                <Input placeholder={content.contact.form.company} {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="message"
          render={({ field }) => (
            <FormItem>
              <FormLabel>{content.contact.form.message}</FormLabel>
              <FormControl>
                <Textarea placeholder={content.contact.form.message} className="min-h-32" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <Button type="submit" size="lg" className="w-full bg-accent text-accent-foreground hover:bg-accent/90">
          {content.contact.form.submit}
        </Button>
      </form>
    </Form>
  );
}
